/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef DEBUG_H
#define DEBUG_H
/*----------------------------------------------------------------------------*/
/**
  @file
  @brief
  This file contains functions, variables and values to control the amount of
  debug messages given by this package.

  @details
  At startup (modprobe) of this package, the variable debug_level can be set to
  a value:
    modprobe elomax debug_level=3
  This pacakge will set its amount of outputted debug-messages as dictated by
  this variable.
  Valid values for this variable are @ref DBG_LVL_NOTHING, @ref DBG_LVL_ERROR,
  @ref DBG_LVL_SUMRY, @ref DBG_LVL_INFO, @ref DBG_LVL_DEBUG. When no (valid)
  value is given during the modprobe command, value @ref DBG_LVL_SUMRY is
  assumed.
**/
/*----------------------------------------------------------------------------*/
#include <linux/kernel.h>
#include "this-module.h"
/*----------------------------------------------------------------------------*/
#ifndef OUR_MODULE
  #warning( "OUR_MODULE is not defined" )
#endif
/*----------------------------------------------------------------------------*/
/**
  @brief
  The debug level of this package.

  @details
  The higher the value in this variable, the more debug messaging is displayed.
  This variable is mapped by udev to /sys/module/elomax/parameters/debug_level.
**/
extern int debug_level;
/*----------------------------------------------------------------------------*/
/**
  @brief
  Outputs a single debug message.

  @details
  The string outputted is displayed in /var/log/syslog and dmesg.
**/
#define debug_msg( format, arg... ) \
  printk( KERN_DEBUG OUR_MODULE ": " format, ##arg )
/*----------------------------------------------------------------------------*/
/**
  @brief
  Debug level nothing: no messages are displayed.

  @details
**/
#define DBG_LVL_NOTHING     0
/**
  @brief
  Debug level error: only error messages are displayed.

  @details
**/
#define DBG_LVL_ERROR       1
/**
  @brief
  Debug level summary: error and connect/disconnect, open/close messages are
  displayed.

  @details
**/
#define DBG_LVL_SUMRY       2
/**
  @brief
  Debug level info: error, connect/disconnect, open/close messages and
  transferred data are displayed.

  @details
**/
#define DBG_LVL_INFO        3
/**
  @brief
  Debug level debug: all messages are displayed, error, connect/disconnect,
  open/close, transferred data and full function debug messages.

  @details
**/
#define DBG_LVL_DEBUG       4
/*----------------------------------------------------------------------------*/
/**
  @brief
  Shortcut to output a debug-message for levels error and above.

  @details
**/
#define DBG_ERROR( format, arg... ) \
  if( DBG_LVL_ERROR <= debug_level ) \
    debug_msg( "%s: " format, __FUNCTION__, ##arg )

/**
  @brief
  Shortcut to output a debug-message for levels summary and above.

  @details
**/
#define DBG_SUMRY( format, arg... ) \
  if( DBG_LVL_SUMRY <= debug_level ) \
    debug_msg( "%s: " format, __FUNCTION__, ##arg )

/**
  @brief
  Shortcut to output a debug-message for levels info and above.

  @details
**/
#define DBG_INFO( format, arg... ) \
  if( DBG_LVL_INFO <= debug_level ) \
    debug_msg( "%s: " format, __FUNCTION__, ##arg )

/**
  @brief
  Shortcut to output a debug-message for levels debug and above.

  @details
**/
#define DBG_DEBUG( format, arg...) \
  if( DBG_LVL_DEBUG <= debug_level ) \
    debug_msg( "%s: " format, __FUNCTION__, ##arg )
/*----------------------------------------------------------------------------*/
/**
  @brief
  Shortcut to output a debug-message when a function is entered.

  @details
**/
#define debug_enter   DBG_DEBUG( "enter\n" )
/**
  @brief
  Shortcut to output a debug-message when a function is left.

  @details
**/
#define debug_leave   DBG_DEBUG( "leave\n" )

/**
  @brief
  Shortcut to output a debug-message when a 'not ok' situation is detected.

  @details
**/
#define debug_err( ok, format, arg... ) \
  if( ! ok ) DBG_ERROR( format, ##arg )
/*----------------------------------------------------------------------------*/
/**
  @brief
  Show the content of the debug_level variable in human readable form (with
  explanation).

  @details
**/
void
debug_show_level( void );
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
