/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef ELOMAX_H
#define ELOMAX_H
/*----------------------------------------------------------------------------*/
/**
  @file
  @brief
  This file defines the interface between the user application and this Elomax
  Driver package.

  @details
  The interface is accessible via the device file. This file is automatically
  created (via udev) when this driver is loaded and accepts control over a
  connected unit.

  The user application can use the <fcntl.h> system library to open(), read()
  and/or write() and close() the device file. The file is named according
  @ref ELOMAX_DEVICE_FILE.
**/
/*----------------------------------------------------------------------------*/
/**
  @brief
  The device file for an elomax unit, the interface for an user application to
  talk to an unit.

  @details
  This 'file' is created by udev, on request of the usb-core, when an elomax
  unit is connected to the usb bus. The %%d is replaced with an integer number,
  to have an unique file for every attached unit.

  When the unit is disconnected, this 'file' is removed by udev.

  A user application uses this file to access the unit. This file allows certain
  types of reads and writes. The reads allowed are:
  - of type @ref elomax_bare_data, to read the bare data from the unit,
  - of type elomax_status_data, to read the bare data from the unit and the
    status of the unit,
  - of type elomax_monitor_data, to read the bare data from the unit, the
    status of the unit and the bare data most recently written to the unit,
  - of type elomax_id_data, to read the type and connection of the unit.

  Only one type of write is allowed on this file:
  - of type @ref elomax_bare_data, to write the bare data to the unit.
**/
#define ELOMAX_DEVICE_FILE           "/dev/elomax%d"
/*----------------------------------------------------------------------------*/
/**
  @brief
  The usb vendor-id of the elomax unit.

  @details
**/
#define ELOMAX_VENDOR_ID             0x07a0

/**
  @brief
  The usb product-id of an elomax iosolution unit.

  @details
  This value is used to identify an unit of type iosolution. It is returned in
  the elomax_id_data::type field in an elomax_id_data typed read (see also
  @ref ELOMAX_DEVICE_FILE).
**/
#define ELOMAX_IOSOLUTION_ID         0x1001
/**
  @brief
  The usb product-id of an elomax aninsolution unit.

  @details
  This value is used to identify an unit of type aninsolution. It is returned in
  the elomax_id_data::type field in an elomax_id_data typed read (see also
  @ref ELOMAX_DEVICE_FILE).
**/
#define ELOMAX_ANINSOLUTION_ID       0x1002
/**
  @brief
  The usb product-id of an elomax fansolution unit.

  @details
  This value is used to identify an unit of type fansolution. It is returned in
  the elomax_id_data::type field in an elomax_id_data typed read (see also
  @ref ELOMAX_DEVICE_FILE).
**/
#define ELOMAX_FANSOLUTION_ID        0x1003

/**
  @brief
  The usb product-id of an unknown elomax unit.

  @details
  This value is used to indicate a failure while retrieving the type of elomax
  unit. It is returned in the elomax_id_data::type filed in an elomax_id_data
  typed read when the retrieving failed (see also @ref ELOMAX_DEVICE_FILE).
**/
#define ELOMAX_UNKNOWN_ID            0x0000
/*----------------------------------------------------------------------------*/
/**
  @brief
  The data-type to use for 'bare' communication with an elomax unit (see also
  @ref ELOMAX_DEVICE_FILE).

  @details
  The format of the 8 data bytes differs per type of elomax unit, see the
  elomax documentation for the details.
**/
struct elomax_bare_data
{
  /**
    @brief
    The bytes containing the actual 'bare' data.

    @details
  **/
  unsigned char  byte[ 8 ];
};
/**
  @brief
  The data-type to use for 'bare' communication with an elomax unit (see also
  @ref ELOMAX_DEVICE_FILE).

  @details
  The format of the 8 data bytes differs per type of elomax unit, see the
  elomax documentation for the details.
**/
typedef struct elomax_bare_data  elomax_bare_data;

/**
  @brief
  Status bit which is set when the read is announced successfull.

  @details
  Use this value to mask the status word.

  The read consists of two steps. First, a read is announced. This is the driver
  appointing some memory to catch the incoming data of the unit. When this
  appointing of memory, announcing the read, is successfull, this status bit is
  set.

  The second step of the read is executed when data of the unit is received and
  the appointed memory is filled. The succesfullness of that step is indicated
  by @ref ELOMAX_READ_RECEIVED.
**/
#define ELOMAX_READ_ANNOUNCED     0x01
/**
  @brief
  Status bit which is set when data from the unit is successfull read.

  @details
  Use this value to mask the status word.

  The read consists of two steps. First, the driver
  appoints some memory to catch the incoming data of the unit. The
  successfullness of this appointing is indicated by @ref
  ELOMAX_READ_ANNOUNCED.

  The second step of the read is executed when data of the unit is received and
  the appointed memory is filled. The succesfullness of that step is indicated
  by this bit.
**/
#define ELOMAX_READ_RECEIVED      0x02
/**
  @brief
  Status bit which is set when at least once data is written to the unit.

  @details
  Use this value to mask the status word.

  For most of the elomax solution units, part of the written data is
  configuration of the unit (eg pull-up resistors on or off, sink current
  level). Since the correct functioning of the unit depends on this
  configuration, the user may want to know if a write is at least once
  succesfully completed.

  This status bit indicates that at least one write is sucesfully completed.
**/
#define ELOMAX_WRITE_ONCEDONE     0x04
/**
  @brief
  Status bit which is set when the last write to the unit was succesfull.

  @details
  Use this value to mask the status word.

  This bit is set when the last write to the unit is completed succesfull.
**/
#define ELOMAX_WRITE_DONE         0x08
/**
  @brief
  The data-type in which the status of the unit is returned.

  @details
  This variable-type holds information about the read/write status of the unit.
  It is a bit-mask, consisting of:
  - @ref ELOMAX_READ_ANNOUNCED, to show the announcement of the read
  - @ref ELOMAX_READ_RECEIVED, to show that data is received
  - @ref ELOMAX_WRITE_ONCEDONE, to show that the unit is 'configured'
  - @ref ELOMAX_WRITE_DONE, to show that the last write was succesfull

  Use macro's @ref ELOMAX_IS_STATUS to read the status of the unit. The macro
  @ref ELOMAX_SET_STATUS is internally used by the driver to set the status.
**/
typedef unsigned char   elomax_status_word;

/**
  @brief
  Macro to read variables of type @ref elomax_status_word.

  @details
  This macro facilitates the reading of a status of the unit, by 'casting' the
  requested bit to a boolean value.

  @param[ in ]  statusword
  The statusword to be investigated. This variable must be of type
  @ref elomax_status_word.

  @param[ in ]  statusbit
  The statusbit to check. This must be one of:
  - @ref ELOMAX_READ_ANNOUNCED,
  - @ref ELOMAX_READ_RECEIVED,
  - @ref ELOMAX_WRITE_ONCEDONE or
  - @ref ELOMAX_WRITE_DONE.

  @return
  The boolean equivalent of the requested bit in the status word:
  - 1, when the bit is set
  - 0, when the bit is reset
**/
#define ELOMAX_IS_STATUS( statusword, statusbit )                              \
  ( 0 != ( statusword & ( statusbit ) ) ? 1 : 0 )
/**
  @brief
  Macro to write variables of type @ref elomax_status_word.

  @details
  This macro is used by the driver internally to set the status of the unit. Do
  not use this macro in an user application.

  @param[ in ]  statusword
  The statusword to be written. This variable must be of type
  @ref elomax_status_word.

  @param[ in ]  statusbit
  The statusbit to check. This must be one of:
  - @ref ELOMAX_READ_ANNOUNCED,
  - @ref ELOMAX_READ_RECEIVED,
  - @ref ELOMAX_WRITE_ONCEDONE or
  - @ref ELOMAX_WRITE_DONE.

  @param[ in ]  value
  The value to set the requested bit to. When zero, the bit is reset, when
  unequal zero, the bit is set.
**/
#define ELOMAX_SET_STATUS( statusword, statusbit, value )                      \
  statusword =                                                                 \
    ( statusword & ( ~ ( statusbit ) ) ) | ( value != 0 ? ( statusbit ) : 0 )
/*----------------------------------------------------------------------------*/
/**
  @brief
  The data-type to read the values and the status of the unit.

  @details
**/
struct elomax_status_data
{
  /**
    @brief
    The values read of the unit.

    @details
  **/
  elomax_bare_data    read;
  /**
    @brief
    The status of the unit.

    @details
  **/
  elomax_status_word  status;
};
/**
  @brief
  The data-type to read the values and the status of the unit.

  @details
**/
typedef struct elomax_status_data   elomax_status_data;

/**
  @brief
  The data-type to read the values, the status and the values recently written
  to the unit.

  @details
**/
struct elomax_monitor_data
{
  /**
    @brief
    The values read of the unit.

    @details
  **/
  elomax_bare_data    read;
  /**
    @brief
    The status of the unit.

    @details
  **/
  elomax_status_word  status;
  /**
    @brief
    The values most recently written to the unit.

    @details
  **/
  elomax_bare_data    write;
};
/**
  @brief
  The data-type to read the values, the status and the values recently written
  to the unit.

  @details
**/
typedef struct elomax_monitor_data  elomax_monitor_data;
/*----------------------------------------------------------------------------*/
/**
  @brief
  The data-type to read the type of unit.

  @details
  The returned value identifies the type of solution:
  - value @ref ELOMAX_IOSOLUTION_ID for an iosolution,
  - value @ref ELOMAX_ANINSOLUTION_ID for an aninsolution,
  - value @ref ELOMAX_FANSOLUTION_ID for a fansolution, or
  - value @ref ELOMAX_UNKNOWN_ID when something went wrong retrieving the type
    of solution connected.
**/
typedef unsigned int  elomax_solution_type;
/*----------------------------------------------------------------------------*/
/**
  @brief
  The data-type to read the connection of the unit.

  @details
  It is a zero terminated string, uniquely identifying where the unit in the
  usb-tree is connected. It is consistent over reboot and only changes when
  the unit is rewired in the usb-tree.
**/
typedef char  elomax_connection_type[ 255 ];
/*----------------------------------------------------------------------------*/
/**
  @brief
  The data-type to read the identification and connection of the unit.

  @details
**/
struct elomax_id_data
{
  /**
    @brief
    The type of unit.

    @details
  **/
  elomax_solution_type    type;
  /**
    @brief
    The connection-point of the unit.
  **/
  elomax_connection_type  connection;
};
/**
  @brief
  The data-type to read the identification and connection of the unit.

  @details
**/
typedef struct elomax_id_data  elomax_id_data;
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
