/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef ELOMAXDRIVER_PRIV_H
#define ELOMAXDRIVER_PRIV_H
/*----------------------------------------------------------------------------*/
/**
  @file
  @brief
  This file contains the functions to recognize an elomax unit when it is
  connected and disconnected from the usb-bus.

  @details
  The elomax_driver_probe() and elomax_driver_disconnect() functions in this
  file are via the 'driver table' (see @ref elomaxdriver.h) communicated to the
  usb-core. The usb-core calls these functions when an elomax driver is
  connected and disconnected from the usb-core.
**/
/*----------------------------------------------------------------------------*/
#include <linux/usb.h>
/*----------------------------------------------------------------------------*/
/**
  @brief
  This function creates an instance for the just connected elomax unit.

  @details
  This function is called by the usb-core when an elomax unit is connected to
  the usb-bus. It checks if the connected unit is really an elomax unit, then
  calls elomax_instance_construct() to create an instance for this unit. This
  instance is then attached to the @e struct @e usb_interface via @e
  usb_set_intfdata().

  @param[ in ]  interface
  The data-structure in which the usb-core keeps its administration for this
  unit.

  @param[ in ]  id
  Additional information the usb-core supplies about the probed device.

  @return
  The successfullness of the creation of the instance.
**/
static int
elomax_driver_probe(
  struct usb_interface        *interface,
  const struct usb_device_id  *id );
/*----------------------------------------------------------------------------*/
/**
  @brief
  This function destroys the instance for the just disconnected elomax unit.

  @details
  This function is called by the usb-core when an elomax unit is disconnected
  from the usb-bus. It uses the supplied @e struct @e usb_interface to refind
  the attached instance (via usb_get_intfdata()), destroys this instance by
  calling elomax_instance_destruct() and resets the pointer from the interface
  to NULL (via usb_set_intfdata()).

  @param[ in ]  interface
  The interface which is disconnected from the usb-bus.
**/
static void
elomax_driver_disconnect( struct usb_interface *interface );
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
