/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#include "bool.h"
#include "debug.h"
#include "elomax.h"
#include "elomaxinstance.h"
#include "elomaxdriver-priv.h"
#include "elomaxdriver.h"
/*----------------------------------------------------------------------------*/
static struct usb_device_id
elomax_usbid_table[] =
{
  { USB_DEVICE( ELOMAX_VENDOR_ID, ELOMAX_IOSOLUTION_ID ) },
  { USB_DEVICE( ELOMAX_VENDOR_ID, ELOMAX_ANINSOLUTION_ID ) },
  { USB_DEVICE( ELOMAX_VENDOR_ID, ELOMAX_FANSOLUTION_ID ) },
  { }
};

MODULE_DEVICE_TABLE( usb, elomax_usbid_table );
/*----------------------------------------------------------------------------*/
static struct usb_driver
elomax_usbdriver_table =
{
  name:                  "elomax",
  probe:                 elomax_driver_probe,
  disconnect:            elomax_driver_disconnect,
  id_table:              elomax_usbid_table
};
/*----------------------------------------------------------------------------*/
int
elomax_driver_construct( void )
{
  int result = 0;

  debug_enter;

  DBG_DEBUG( "register driver to usb-core\n" );
  result = usb_register( &elomax_usbdriver_table );
  debug_err(
    0 == result,
    "register driver to usb-core failed with result %d\n",
    result );

  debug_leave;

  return result;
}
/*----------------------------------------------------------------------------*/
void
elomax_driver_destruct( void )
{
  debug_enter;

  DBG_DEBUG( "unregister driver from usb-core\n" );
  usb_deregister( &elomax_usbdriver_table );

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static int
elomax_driver_probe(
  struct usb_interface       *interface,
  const struct usb_device_id *id )
{
  struct usb_device  *device = interface_to_usbdev( interface );
  BOOL ok = TRUE;
  struct elomax_instance *this_instance = NULL;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "check vendor-id and product-id\n" );
    ok =
      ( ELOMAX_VENDOR_ID == device->descriptor.idVendor ) &&
      (
        ( ELOMAX_IOSOLUTION_ID == device->descriptor.idProduct ) ||
        ( ELOMAX_ANINSOLUTION_ID == device->descriptor.idProduct ) ||
        ( ELOMAX_FANSOLUTION_ID == device->descriptor.idProduct )
      );
    debug_err( ok, "check vendor-id and product-id failed\n" );
    }

  if( ok )
  {
    DBG_DEBUG( "construct an instance for this interface\n" );
    this_instance = elomax_instance_construct( interface );
    ok = NULL != this_instance;
    debug_err( ok, "construct an instance for this interface failed\n" );
  }

  if( ok )
  {
    DBG_DEBUG( "save instance in interface\n" );
    usb_set_intfdata( interface, this_instance );
  }

  debug_leave;

  return ok ? 0 : -ENODEV;
}
/*----------------------------------------------------------------------------*/
static void
elomax_driver_disconnect( struct usb_interface *interface )
{
  struct elomax_instance *this_instance = NULL;

  debug_enter;

  DBG_DEBUG( "find instance and tell usb-core we're not servicing anymore\n" );
  this_instance = usb_get_intfdata( interface );
  usb_set_intfdata( interface, NULL );

  DBG_DEBUG( "destruct this instance\n" );
  elomax_instance_destruct( this_instance );

  debug_leave;
}
/*----------------------------------------------------------------------------*/
struct elomax_instance *
elomax_driver_instance_from_minor( unsigned int minor )
{
  struct usb_interface *interface = NULL;
  struct elomax_instance *instance = NULL;

  debug_enter;

  DBG_DEBUG( "ask usb-core for interface\n" );
  interface = usb_find_interface( &elomax_usbdriver_table, minor );
  DBG_DEBUG( "usb-core told interface at %p\n", interface );
  DBG_DEBUG( "ask interface for instance\n" );
  instance = usb_get_intfdata( interface );
  DBG_DEBUG( "interface told instance at %p\n", instance );

  debug_leave;
  return instance;
}
/*----------------------------------------------------------------------------*/
