/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef ELOMAXDRIVER_H
#define ELOMAXDRIVER_H
/*----------------------------------------------------------------------------*/
/**
  @file
  @brief
  This file contains the code to connect this package to the usb-core, so that
  it is informed about the connection/disconnection of elomax units.

  @details
  When this driver is loaded into the kernel, it announces its 'driver table'
  (a @e struct @e usb_driver) to the usb-core by calling @e usb_register().
  This table contains amongst other the VendorId, ProductId pairs of the units
  serviced, together with the functions to call upon connection and
  disconnection of such a unit.

  Upon connection of an elomax unit, the usb-core creates a @e struct
  @e usb_interface for that unit. This struct contains usb-core specific
  information about the state of the unit. With this struct, the usb-core calls
  the elomax_driver_probe() function.

  The elomax_driver_probe() function creates an instance for this connected
  unit (by calling elomax_instance_construct()). In this struct, this package
  keeps the unit specific data. Since this struct is an extension to the
  usb-core's struct, the probe function connects the instance to the
  usb-core's struct by calling @e usb_set_intfdata().

  The opposite occurs when the usb-core detects the disconnection of an unit.
  By the 'driver table' the usb-core knows to call the
  elomax_driver_disconnect() function. It supplies the @e struct
  @e usb_interface as a parameter.

  With this parameter, the elomax_driver_disconnect() can find its own
  @e elomax_instance and destructs it (by calling elomax_instance_destruct()).
  It sets the reference in the @e struct @e usb_interface to NULL via the
  @e usb_set_intfdata() function.

  For every connected elomax unit, the usb-core maintains a @e struct
  @e usb_interface and this package maintains an elomax_instance. The @e struct
  @e usb_interface keeps a pointer to the @e struct @e elomax_instance, which is
  set via @e usb_setintfdata().

  After connection of a unit, the usb-core arranges the following:
  - the usb-core tells the kernel that the elomax module is in use, so that
    it cannot be unloaded
  - the usb-core tells @e udev that an unit is connected. @e udev will create
    a device-file with the appropriate minor, so that the user can connect to
    the elomax driver.
**/
/*----------------------------------------------------------------------------*/
#include <linux/usb.h>
/*----------------------------------------------------------------------------*/
/**
  @brief
  Announce this driver to the usb-core, so that it is informed about the
  connection and disconnection of elomax units on the usb-bus.

  @details
  This function calls @e usb_register() with its 'driver table' as parameter.
  Since the driver table contains references to the elomax_driver_probe() and
  elomax_driver_disconnect() function, and to the 'id table' containing the
  VendorId, ProductId pairs of the units which this driver can service, this
  driver gets informed about the connection/disconnection of these units.

  @return
  The successfullness of announcing this driver.
**/
int
elomax_driver_construct( void );
/*----------------------------------------------------------------------------*/
/**
  @brief
  Tell the usb-core about the removal of this driver, so that it is no longer
  informed about the connection and disconnection of elomax units on the
  usb-bus.

  @details
  This function calls @e usb_deregister(), so that the usb-core knows that this
  driver is no longer in operation.
**/
void
elomax_driver_destruct( void );
/*----------------------------------------------------------------------------*/
struct elomax_instance;
/**
  @brief
  Tell the instance belonging to a device-file with the given minor.

  @details
  This function calls @e usb_find_interface() to find the @e struct
  @e usb_interface belonging to the given minor. This interface can tell the
  corresponding elomax_instance via @e usb_get_intfdata.

  There is NO error checking in this function.

  @param[ in ]  minor
  The minor of the file for which the corresponding elomax_instance must be
  found.

  @return
  - the elomax_instance belonging to the device file with the given minor, or
  - NULL when something went wrong.
**/
struct elomax_instance *
elomax_driver_instance_from_minor( unsigned int minor );
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
