/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef ELOMAXINSTANCE_PRIV_H
#define ELOMAXINSTANCE_PRIV_H
/*----------------------------------------------------------------------------*/
/**
  @file
  @brief
  This file contains the functions to manipulate every connected elomax unit
  from user space.

  @brief
  The functions elomax_instance_read_complete() and
  elomax_instance_write_complete() are 'internal' functions to the instance.
  They are called by the usb-core when an urb read resp. write is complete.

  The other functions in this file are called when an user application
  manipulates the connected elomax unit via its device file. The user can open,
  read, write and close the device file.

  When the user does so, the kernel calls elomax_instance_open(),
  elomax_instance_read(), elomax_instance_write() respectively
  elomax_instance_close(). The kernel knows to call these functions, because
  they are pointed to by the 'file operation table', which via the 'class
  driver table' was made known to the usb-core, when the unit was connected
  to the usb-bus (and the 'class driver table' announced via
  elomax_instance_construct()).
**/
/*----------------------------------------------------------------------------*/
#include <linux/usb.h>
#include "elomax.h"
/*----------------------------------------------------------------------------*/
/**
  @brief
  Maintains the status of the elomax unit it represents.

  @details
  This struct holds:
  - instance_smartptr, a kernel smartpointer, reference counting the pointers
    to this instance
  - instance_protect, the lock guarding this instance against simultaneous
    access
  - interface, the @e struct @e usb_interface this instance is attached to
  - and others...
**/
struct elomax_instance
{
  /* instance */
  /**
    @brief
    Keeps the reference count of this instance.

    @details
    The instance is constructed dynamically, upon connection of a unit by
    elomax_instance_construct(). A reference to it (pointer) is kept at
    multiple places:
    - in the @e struct @e usb_interface's intfdata by the usb-core, so that this
      package can refind its own data
    - in the read and write urb's context, so that the completion functions can
      refind their data
    - in the @e struct @e file opened by the user, so that following file
      operations can refind their data

    This smartptr (reference count) keeps the number of pointers stored, and
    is incremented/decremented by every copy/release of the pointer. The last
    pointer release (when the reference counter reaches zero) triggers the
    free-ing of memory for this instance.
  **/
  struct kref           instance_smartptr;
  /**
    @brief
    The lock on the instance to protect it against simultaneous multi-threaded
    access.

    @details
    The instance is operated by multiple threads:
    - there is a thread from the usb-core to inform the instance about
      disconnection of the unit from the usb-bus
    - there is a thread from the usb-core to inform the instance about
      completion of an urb (possibly the same thread as above)
    - there is a thread from the user executing a file operation

    Each thread wanting access to the instance, locks (@e spin_lock()) the
    instance before accessing it and unlocks (@e spin_unlock()) it afterwards.
    The lock may temporarily delay the thread (because another thread holds the
    lock), however it will not put the waiting thread to sleep. The functions in
    this package assure that the lock is held as short as possible.
  **/
  spinlock_t            instance_protect;
  /**
    @brief
    The usb-interface under which this unit is known to the usb-core.

    @details
    The usb-core remembers every connected unit as an interface. This package
    attaches its 'personal' data for an elomax unit to this interface (upon
    connection of the unit to the usb-bus). This way, when the usb-core signals
    something for an interface, this package can refind its 'personal' data.

    The other way around, is the instance attached to the file. So every
    file operation can refind the 'personal' data for the elomax unit. To be
    able to submit requests to the usb-core, this field allows to refind the
    usb-interface belonging to this instance.
  **/
  struct usb_interface *interface;
  /**
    @brief
    Holds the most recent read and written data from and to the unit.

    @details
    The recent communication with the unit is stored in this field. It maintains
    the recently read bytes from the unit, the recently written bytes to the
    unit and the status of these read and write. This field is updated by the
    completion functions elomax_instance_read_copmlete() and
    elomax_instance_write_complete().
  **/
  struct elomax_monitor_data
                        recently;          /* most recently read/written */
  /* usb read */
  /**
    @brief
    The urb used to receive information from the unit.

    @details
    This urb is reserved upon construction of the instance. It then is
    submitted to the usb-core to receive data from the unit. When the usb-core
    receives data from the unit, it calls the completion-function
    elomax_instance_read_complete() which copies the data from this urb to the
    elomax_instance::recently field and resubmits this read_urb for receiving
    the next data of the unit.
  **/
  struct urb           *read_urb;          /* the urb used for reading */
  /* usb write */
  /**
    @brief
    The urb used to send information to the unit.

    @details
    This urb is reserved upon construction of the instance. It is then filled
    with the data, when the user application writes to the device-file, so
    executing elomax_instance_write(). When the write completes, the function
    elomax_instance_write_complete() remembers the written data in the
    elomax_instance::recently field and marks the urb ready for reuse by freeing
    the semaphore elomax_instance::write_semaphore.
  **/
  struct urb           *write_urb;         /* the urb used for writing */
  /**
    @brief
    Holds of (puts to sleep) simultaneous writers, so that only one write at a
    time is active.

    @details
    This semaphore is initialized to 1 and guards the write urb of this
    instance. The user's file operation elomax_instance_write() claims the
    semaphore. This semaphore is released when the write urb (submitted by the
    write who claimed the semaphore) finishes (by function
    elomax_instance_write_complete()).

    Note that a user's write may be delayed (put to sleep) because another
    user thread holds the semaphore (as long as its write is not finished, that
    is, the data transferred to the unit [or timeout]).
  **/
  struct semaphore      write_semaphore;
  /**
    @brief
    The setup-packet used to write data to the unit.

    @details
    The elomax unit uses an usb control-write to transfer data from the host
    to the unit. Such a control-write consists of a setup-packet (host to
    device), a data-packet (host to device) and an acknowledgement (device to
    host).

    Even though the setup-packet is a const (always contains the same data) it
    is kept per instance, since it must be allocated in contiguous kernel
    memory.
  **/
  unsigned char         write_setup_packet[ 8 ]; /* must be contiguous
                                              kernel memory */
};
/*----------------------------------------------------------------------------*/
/**
  @brief
  When the instance's read urb is finished, this function saves the received
  data and resubmits the read urb for the next incoming data.

  @details
  The read urb is marked with this function as completion function, therefore
  this function is called by the usb-core when the read finishes. This function
  retrieves the instance from the urb's context, checks the urb's status (the
  succesfullness of the read) and then stores the received data in the instance.

  The last action of this function is to submit this read urb again to the
  usb-core (via @e usb_submit_urb), so that this package is continously
  prepared for receiving the latest data of the unit.

  @param[ in ]  urb
  The urb which is just finished. This urb is marked with this function as
  completion-function, hence this function is called. The urb contains the
  instance for which it did the write in its context (placed there upon
  submission of the urb).
**/
static void
elomax_instance_read_complete( struct urb *urb );
/*----------------------------------------------------------------------------*/
/**
  @brief
  When this instance's write urb is finished, this function releases the
  instance's lock (semaphore) to allow the next writer.

  @details
  The instance is retrieved from the urb's context, after which this function
  the status of the urb (succesfullness of the write) checks. It then releases
  (@e up() of the instance's elomax_instance::write_semaphore) to allow the next
  writer.

  @param[ in ]  urb
  The urb which is just finished. This urb is marked with this function as
  completion-function, hence this function is called. The urb contains the
  instance for which it did the write in its context (placed there upon
  submission of the urb).
**/
static void
elomax_instance_write_complete( struct urb *urb );
/*----------------------------------------------------------------------------*/
/**
  @brief
  Deliver data from the device to the user.

  @details
  This read function only wants to deliver certain amounts (and therefore types)
  of data. These data formats are described in @ref elomax.h.

  When the user requested a valid amount (type) of data, this function checks
  if no destruct announcement has been made.

  With all checks ok, it copies the data, gathered by the
  elomax_instance_read_complete() (and elomax_instance_write_complete())
  functions to the user.

  @param[ in ]  file
  The file from which the user wants to read. This file 'points' to our device
  file.

  @param[ in ]  buffer
  The buffer in which the user wants to receive the data. This buffer
  is in user-space, not in kernel-space.

  @param[ in ]  count
  The count of databytes the user wants to receive. This count is used to
  differentiate between the 'types of data' this driver can deliver.

  @param[ in ]  ppos
  The position at which the user wants to start the read. This parameter is
  ignored in this package.

  @return
  - the bytes delivered to the user by this package, or
  - -ENOMEM when something went wrong.
**/
static ssize_t
elomax_instance_read(
  struct file  *file,
  char         *buffer,
  size_t        count,
  loff_t       *ppos );
/*----------------------------------------------------------------------------*/
/**
  @brief
  Deliver data from the user to the device.

  @details
  This write function only accepts one 'data format'. This data format
  for each type of 'solution' is described in @ref elomax.h.

  When the user requested the right amount of data to be written, this function
  requests the instance for permission to write, by @e down() the instance's
  write_semaphore. This may put the calling thread to sleep.

  With permission granted, it checks that the instance is not announced for
  destruction, fills the write-urb with the user's data and sends it to the unit
  (via the usb-core's @e usb_submit_urb()).

  The semaphore is kept until the usb-core calls the
  elomax_instance_write_complete(). That function releases (@e up()) the
  semaphore (this may wakeup a waiting thread).

  @param[ in ]  file
  The file to which the user wants to write, it points to our device file.

  @param[ in ]  buffer
  The data the user wants to write.

  @param[ in ]  count
  The amount of databytes the user wants to write. This parameter is used to
  check the 'data type' for the write.

  @param[ in ]  ppos
  The position at which the user wants to write the data. This parameter is
  ignored by this package.

  @return
  - the bytes delivered by this package to the device, or
  - -ENOMEM when something went wrong.
**/
static ssize_t
elomax_instance_write(
  struct file  *file,
  const char   *buffer,
  size_t        count,
  loff_t       *ppos );
/*----------------------------------------------------------------------------*/
/**
  @brief
  Open the device file, so that the user can read/write to the device.

  @details
  With the minor stored in the inode, this function retrieves the usb
  interface coupled to this device file. This interface is then asked for its
  instance, its attachment (via elomax_driver_instance_from_minor()).

  When found, this instance is coupled to the file (by storing it in the file's
  private_data), so that following reads and writes can find the instance. Also
  the smartptr reference counter is incremented.

  @param[ in ]  inode
  The inode of the file requested for opening (the device file).

  @param[ in ]  file
  The file requested for opening (the device file).

  @return
  The succesfullness of opening:
  - 0 when ok,
  - -ENODEV when not ok.
**/
static int
elomax_instance_open(
  struct inode  *inode,
  struct file   *file );
/*----------------------------------------------------------------------------*/
/**
  @brief
  Close the given file, so that the user no longer can read or write to it.

  @details
  This function clears the file's private_data (filled by
  elomax_instance_open()) and decrements the usage count (smartptr, reference
  counter) and possibly destroys the instance.

  @param[ in ]  inode
  The inode of the device file to be closed.

  @param[ in ]  file
  The device file to be closed.

  @return
  The succesfullness of the closing:
  - 0 when ok,
  - -ENODEV when something went wrong.
**/
static int
elomax_instance_close(
  struct inode  *inode,
  struct file   *file );
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
