/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#include <linux/uaccess.h>
#include "bool.h"
#include "debug.h"
#include "elomaxdriver.h"
#include "elomaxinstance-priv.h"
#include "elomaxinstance.h"
/*----------------------------------------------------------------------------*/
#define STATIC_CAST( type, value )     ( ( type )( value ) )
/*----------------------------------------------------------------------------*/
#define CONCURRENT_WRITES        1
/*----------------------------------------------------------------------------*/
#define HIBYTE( value )      ( ( value ) / 256 )
#define LOBYTE( value )      ( ( value ) % 256 )
#define WORD( value )        ( LOBYTE( value ), HIBYTE( value ) )

#define FILL_SETUP( setup, bmReq, bReq, wVal, wIndex, wLength ) \
  setup[ 0 ] = bmReq;            setup[ 1 ] = bReq;             \
  setup[ 2] = LOBYTE( wVal );    setup[ 3 ] = HIBYTE( wVal );   \
  setup[ 4] = LOBYTE( wIndex );  setup[ 5 ] = HIBYTE( wIndex ); \
  setup[ 6] = LOBYTE( wLength ); setup[ 7 ] = HIBYTE( wLength );
/*----------------------------------------------------------------------------*/
/* the file-operations this instance supports */
static const struct file_operations
elomax_instance_fops_table =
{
  owner:    THIS_MODULE,
  read:     elomax_instance_read,
  write:    elomax_instance_write,
  open:     elomax_instance_open,
  release:  elomax_instance_close
};
/*----------------------------------------------------------------------------*/
static struct usb_class_driver
elomax_class_driver =
{
  name:       ELOMAX_DEVICE_FILE,
  fops:       &elomax_instance_fops_table,
  minor_base:  0
};
/*----------------------------------------------------------------------------*/
static void
elomax_instance_read_complete( struct urb *urb )
{
  BOOL ok = TRUE;
  int submit_status = 0;
  struct elomax_instance *this = NULL;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "retrieve instance from urb\n" );
    this = urb->context;
    DBG_DEBUG( "retrieved instance from urb at %p\n", this );
    ok = NULL != this;
    debug_err( ok, "retrieve instance from urb failed\n" );
  }
  if( ok )
  {
    DBG_DEBUG( "check urb must be instance's read-urb\n" );
    ok = urb == this->read_urb;
    DBG_DEBUG( "urb=%p, instance's read-urb=%p\n", urb, this->read_urb );
    debug_err(
      ok,
      "check urb=%p must be instance's read-urb=%p\n",
      urb,
      this->read_urb );
  }
  if( ok )
  {
    DBG_DEBUG( "check read-urb's status\n" );
    ok = 0 == urb->status;
    DBG_DEBUG( "read-urb's status=%d\n", urb->status );
    debug_err(
      ok,
      "check read-urb's status failed with status=%d\n",
      urb->status );
  }
  if( ok )
  {
    DBG_DEBUG( "check read-urb's read bytes\n" );
    ok = sizeof( elomax_bare_data ) == urb->actual_length;
    DBG_DEBUG( "read-urb read %d bytes\n", urb->actual_length );
    debug_err(
      ok,
      "check read-urb's read bytes failed, need %lu, read %d bytes\n",
      sizeof( elomax_bare_data ),
      urb->actual_length );
  }
  if( ok )
  {
    DBG_INFO(
      "received data=%02x %02x %02x %02x %02x %02x %02x %02x\n",
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 0 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 1 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 2 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 3 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 4 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 5 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 6 ],
      STATIC_CAST( unsigned char *, urb->transfer_buffer )[ 7 ] );
  }
  if( ok )
  {
    int i = 0;

    /* see if we're still alive, not announced_end_of_life */
    spin_lock( &this->instance_protect );
    /*
      and keep the lock,
      so no one can announce us end-of-life while we're busy
    */
    DBG_DEBUG( "remember read bytes for user\n" );
    for( i = 0; i < sizeof( elomax_bare_data ); ++i )
    {
      this->recently.read.byte[ i ] =
        STATIC_CAST( unsigned char *, urb->transfer_buffer )[ i ];
    }
    DBG_INFO(
      "read data=0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",
      this->recently.read.byte[ 0 ], this->recently.read.byte[ 1 ],
      this->recently.read.byte[ 2 ], this->recently.read.byte[ 3 ],
      this->recently.read.byte[ 4 ], this->recently.read.byte[ 5 ],
      this->recently.read.byte[ 6 ], this->recently.read.byte[ 7 ] );
    DBG_DEBUG( "status read.byte-received = 1\n" );
    ELOMAX_SET_STATUS( this->recently.status, ELOMAX_READ_RECEIVED, 1 );
    if( NULL != this->interface )
    {
      DBG_DEBUG( "submit read-urb to the usb-core\n" );
      submit_status = usb_submit_urb( this->read_urb, GFP_ATOMIC );
      ok = 0 == submit_status;
      debug_err(
        ok,
        "submit read-urb to the usb-core failed with status=%d\n",
        submit_status );
    }
    DBG_DEBUG(
      "status read-announced = %d\n",
      ok && ( NULL != this->interface ) );
    ELOMAX_SET_STATUS(
      this->recently.status,
      ELOMAX_READ_ANNOUNCED,
      ok && ( NULL != this->interface ) );
    spin_unlock( &this->instance_protect );
  }

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static void
elomax_instance_write_complete( struct urb *urb )
{
  BOOL ok = TRUE;
  struct elomax_instance *this = NULL;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "retrieve instance from urb\n" );
    this = urb->context;
    DBG_DEBUG( "retrieved instance from urb at %p\n", this );
    ok = NULL != this;
    debug_err( ok, "retrieve instance from urb failed\n" );
  }
  if( ok )
  {
    DBG_DEBUG( "check urb's status\n" );
    ok = 0 == urb->status;
    DBG_DEBUG( "urb's status=%d\n", urb->status );
    debug_err(
      ok,
      "check urb's status failed with status=%d\n",
      urb->status );
  }
  if( ok )
  {
    DBG_DEBUG( "check urb's written bytes\n" );
    ok = sizeof( elomax_bare_data ) == urb->actual_length;
    DBG_DEBUG( "urb wrote %d bytes\n", urb->actual_length );
    debug_err(
      ok,
      "check urb's written bytes failed, wrote %d bytes\n",
      urb->actual_length );
  }
  {
    /* lock unconditionally */
    spin_lock( &this->instance_protect );
  }
  if( ok )
  {
    int i = 0;

    for( i = 0; sizeof( elomax_bare_data ) > i; ++i )
    {
      this->recently.write.byte[ i ] =
        STATIC_CAST( unsigned char *, urb->transfer_buffer )[ i ];
    }
    DBG_INFO(
      "wrote data=0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",
      this->recently.write.byte[ 0 ], this->recently.write.byte[ 1 ],
      this->recently.write.byte[ 2 ], this->recently.write.byte[ 3 ],
      this->recently.write.byte[ 4 ], this->recently.write.byte[ 5 ],
      this->recently.write.byte[ 6 ], this->recently.write.byte[ 7 ] );
  }
  {
    DBG_DEBUG( "status write-done = %d\n", ok );
    ELOMAX_SET_STATUS( this->recently.status, ELOMAX_WRITE_DONE, ok );

    DBG_DEBUG( "status write-once-done = %d\n", ok );
    ELOMAX_SET_STATUS(
      this->recently.status,
      ELOMAX_WRITE_ONCEDONE,
      ok || ELOMAX_IS_STATUS( this->recently.status, ELOMAX_WRITE_ONCEDONE ) );
  }
  {
    /* unlock unconditionally */
    spin_unlock( &this->instance_protect );
  }
  {
    /* allow the next writer unconditionally */
    DBG_DEBUG( "allow the next writer\n" );
    if( NULL != this )
    {
      up( &this->write_semaphore );
    }
  }

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static void
elomax_instance_free_urb_buffer(
  struct elomax_instance *this,
  struct urb **reserved_urb,
  const char *reason )
{
  struct usb_device *device = NULL;

  debug_enter;

  if( NULL != this->interface )
  {
    device = interface_to_usbdev( this->interface );
  }

  if( NULL != *reserved_urb )
  {
    if( NULL != ( *reserved_urb )->transfer_buffer )
    {
      DBG_DEBUG(
        "give back %s-buffer at %p\n",
        reason,
        ( *reserved_urb )->transfer_buffer );
      usb_buffer_free(
        device,
        sizeof( elomax_bare_data ),
        ( *reserved_urb )->transfer_buffer,
        ( *reserved_urb )->transfer_dma );
    }
    DBG_DEBUG(
      "give back %s-urb at %p\n",
      reason,
      *reserved_urb );
    usb_free_urb( *reserved_urb );
  }
  *reserved_urb = NULL;

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static BOOL
elomax_instance_reserve_urb_buffer(
  struct elomax_instance *this,
  struct urb **reserved_urb,
  const char *reason )
{
  struct usb_device *device = NULL;
  BOOL ok = TRUE;

  debug_enter;

  if( NULL != this->interface )
  {
    device = interface_to_usbdev( this->interface );
  }
  *reserved_urb = NULL;
  if( ok )
  {
    DBG_DEBUG( "reserve memory for %s-urb\n", reason );
    *reserved_urb = usb_alloc_urb( 0, GFP_KERNEL );
    ok = NULL != *reserved_urb;
    debug_err( ok, "reserve memory for %s-urb failed\n", reason );
    if( ok )
    {
      DBG_DEBUG(
        "reserved memory for %s-urb at %p\n",
        reason,
        *reserved_urb );
    }
  }
  if( ok )
  {
    DBG_DEBUG( "reserve memory for %s-buffer\n", reason );
    ( *reserved_urb )->transfer_buffer =
      usb_buffer_alloc(
        device,
        sizeof( elomax_bare_data ),
        GFP_KERNEL,
        &( *reserved_urb )->transfer_dma );
    ok = NULL != ( *reserved_urb )->transfer_buffer;
    debug_err( ok, "reserve memory for %s-buffer failed\n", reason );
    if( ok )
    {
      DBG_DEBUG(
        "reserved memory for %s-buffer at %p\n",
        reason,
        ( *reserved_urb )->transfer_buffer );
    }
  }
  if( ok )
  {
    ( *reserved_urb )->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
  }

  if( ! ok )
  {
    DBG_DEBUG( "something failed, clean up\n" );
    elomax_instance_free_urb_buffer( this, reserved_urb, reason );
 }

  debug_leave;

  return ok;
}
/*----------------------------------------------------------------------------*/
static void
elomax_instance_use( struct elomax_instance *this )
{
  debug_enter;

  kref_get( &this->instance_smartptr );

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static void
elomax_instance_free( struct kref *smartptr )
{
  struct elomax_instance *
  this = container_of( smartptr, struct elomax_instance, instance_smartptr );

  debug_enter;

  DBG_DEBUG( "give back memory for read-urb at %p\n", this->read_urb );
  elomax_instance_free_urb_buffer(
    this,
    &this->read_urb,
    "read" );
  DBG_DEBUG( "give back memory for write-urb at %p\n", this->write_urb );
  elomax_instance_free_urb_buffer(
    this,
    &this->write_urb,
    "write" );
  DBG_DEBUG( "give back memory for this instance %p\n", this );
  kfree( this );

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static void
elomax_instance_destroy( struct elomax_instance *this )
{
  debug_enter;
  /* decrement usage count, when we're the last, destroy ourself */
  DBG_DEBUG( "decrement smartptr, free when we're the last\n" );
  kref_put( &this->instance_smartptr, elomax_instance_free );
  debug_leave;
}
/*----------------------------------------------------------------------------*/
struct elomax_instance *
elomax_instance_construct( struct usb_interface *interface )
{
  struct elomax_instance *this = NULL;
  BOOL ok = TRUE;
  struct usb_host_interface *interface_descriptor = interface->cur_altsetting;
  struct usb_endpoint_descriptor *endpoint = NULL;

  debug_enter;

#define READ_IRQ_NR_ENDP       1
#define READ_IRQ_ENDP_ADDR     129
  /* do all our checking */
  if( ok )
  {
    DBG_DEBUG( "check number of endpoints\n" );
    ok = READ_IRQ_NR_ENDP == interface_descriptor->desc.bNumEndpoints;
    debug_err( ok, "check number of endpoints failed\n" );
  }
  if( ok )
  {
    endpoint = &interface_descriptor->endpoint[ 0 ].desc;
    DBG_DEBUG( "check endpoint's properties\n" );
    ok =
      usb_endpoint_is_int_in( endpoint ) &&
      ( sizeof( elomax_bare_data ) ==
          le16_to_cpu( endpoint->wMaxPacketSize ) ) &&
      ( READ_IRQ_ENDP_ADDR == endpoint->bEndpointAddress );
    debug_err(
      ok,
      "check endpoint properties failed, is_irq=%d, datasize=%d, address=%d\n",
      usb_endpoint_is_int_in( endpoint ),
      le16_to_cpu( endpoint->wMaxPacketSize ),
      endpoint->bEndpointAddress );
  }
  /* reserve memory */
  if( ok )
  {
    DBG_DEBUG( "reserve memory for this instance\n" );
    this = kzalloc( sizeof( struct elomax_instance ), GFP_KERNEL );
    DBG_DEBUG( "memory for this instance is at %p\n", this );
    ok = NULL != this;
    debug_err( ok, "reserve memory for this instance failed\n" );
  }
  if( ok )
  {
    DBG_DEBUG( "initialize this instance\n" );
    kref_init( &this->instance_smartptr );
    sema_init( &this->write_semaphore, CONCURRENT_WRITES );
    spin_lock_init( &this->instance_protect );
    /* remember the interface we're working for */
    /* this is also synchronisation to the io-thread that we're not disconn. */
    this->interface = interface;
    this->recently.status = 0x0;
  }
  if( ok )
  {
    DBG_DEBUG( "reserve read-urb\n" );
    this->read_urb = NULL;
    ok =
      elomax_instance_reserve_urb_buffer(
        this,
        &this->read_urb,
        "read" );
  }
  if( ok )
  {
    DBG_DEBUG( "reserve write-urb\n" );
    this->write_urb = NULL;
    ok =
      elomax_instance_reserve_urb_buffer(
        this,
        &this->write_urb,
        "write" );
  }
  if( ok )
  {
    int register_status = 0;
    /* and announce our existance at the usb-core */
    DBG_DEBUG( "register interface to usb-core\n" );
    register_status = usb_register_dev( interface, &elomax_class_driver );
    DBG_DEBUG(
      "registered interface with status %d to minor %d\n",
      register_status,
      interface->minor );
    ok = 0 == register_status;
    debug_err(
      ok,
      "register interface to usb-core failed with status=%d\n",
      register_status );
  }

  if( ok )
  {
    /* start reading the instance */
    struct usb_device *device = interface_to_usbdev( interface );
    int pipe = usb_rcvintpipe( device, READ_IRQ_ENDP_ADDR );
    int submit_status = 0;

    DBG_DEBUG( "fill read-urb for submission\n" );
    usb_fill_int_urb(
      this->read_urb,                 /* urb to fill */
      device,                         /* device to send it to */
      pipe,                           /* pipe to send it over */
      this->read_urb->transfer_buffer,/* data buffer to send */
      sizeof( elomax_bare_data ),     /* data buffer size */
      elomax_instance_read_complete,  /* func to call at completion */
      this,                           /* context to store in the urb */
      endpoint->bInterval );          /* interval to send it at */

    /*
    http://gnugeneration.com/books/linux/2.6.20/usb/re36.html
    
    The general rules for how to decide which mem_flags to use are the same as
    for kmalloc. There are four different possible values; GFP_KERNEL, GFP_NOFS,
    GFP_NOIO and GFP_ATOMIC.
    
    GFP_NOFS is not ever used, as it has not been implemented yet.
    
    GFP_ATOMIC is used when
      (a) you are inside a completion handler, an interrupt, bottom half,
          tasklet or timer, or
      (b) you are holding a spinlock or rwlock (does not apply to semaphores),
          or
      (c) current->state != TASK_RUNNING, this is the case only after you've
          changed it.
    
    GFP_NOIO is used in the block io path and error handling of storage devices.
    
    All other situations use GFP_KERNEL.
    
    Some more specific rules for mem_flags can be inferred, such as
      (1) start_xmit, timeout, and receive methods of network drivers must use
          GFP_ATOMIC (they are called with a spinlock held);
      (2) queuecommand methods of scsi drivers must use GFP_ATOMIC (also called
          with a spinlock held);
      (3) If you use a kernel thread with a network driver you must use
          GFP_NOIO, unless (b) or (c) apply;
      (4) after you have done a down you can use GFP_KERNEL, unless (b) or (c)
          apply or your are in a storage driver's block io path;
      (5) USB probe and disconnect can use GFP_KERNEL unless (b) or (c) apply;
          and
      (6) changing firmware on a running storage or net device uses GFP_NOIO,
          unless b) or c) apply 
    */
    DBG_DEBUG( "submit read-urb to the usb-core\n" );
    submit_status = usb_submit_urb( this->read_urb, GFP_KERNEL );
    ok = 0 == submit_status;
    debug_err(
      ok,
      "submission read-urb to the usb-core failed with status=%d\n",
      submit_status );
    DBG_DEBUG( "status read-announced = %d\n", ok );
    ELOMAX_SET_STATUS( this->recently.status, ELOMAX_READ_ANNOUNCED, ok );
  }
  if( ok )
  {
    DBG_SUMRY( "connected instance %p for minor %d\n", this, interface->minor );
    DBG_DEBUG( "construct to interface at %p\n", this->interface );
    DBG_DEBUG(
      "construct to device at %p\n", interface_to_usbdev( interface ) );
  }

  /* cleanup when something fails */
  if( ! ok )
  {
    DBG_DEBUG( "something failed, cleanup\n" );
    if( NULL != this )
    {
      elomax_instance_destroy( this );
      this = NULL;
    }
  }

  return this;
}
/*----------------------------------------------------------------------------*/
void
elomax_instance_destruct( struct elomax_instance *this )
{
  int minor = this->interface->minor;
  unsigned long irqflags = 0L;

  debug_enter;
  DBG_SUMRY( "disconnected instance %p for minor %d\n", this, minor );

  DBG_DEBUG( "unregister interface from usb-core\n" );
  usb_deregister_dev( this->interface, &elomax_class_driver );
  DBG_DEBUG( "unregistered minor %d from usb-core\n", minor );

  /* kill I/O in progress and prevent more I/O from starting */
  DBG_DEBUG( "stop i/o in progress and prevent starting i/o\n" );
  spin_lock_irqsave( &this->instance_protect, irqflags );
  this->interface = NULL;
  usb_kill_urb( this->read_urb );
  spin_unlock_irqrestore( &this->instance_protect, irqflags );

  DBG_DEBUG( "destroy this instance\n" );
  elomax_instance_destroy( this );

  debug_leave;
}
/*----------------------------------------------------------------------------*/
static int
elomax_instance_read_io_data(
  struct elomax_instance *this,
  char                   *buffer,
  size_t                  count )
{
  int retval = copy_to_user( buffer, this->recently.read.byte, count );

  DBG_INFO(
    "user read read-bytes=%02x %02x %02x %02x %02x %02x %02x %02x\n",
    this->recently.read.byte[ 0 ], this->recently.read.byte[ 1 ],
    this->recently.read.byte[ 2 ], this->recently.read.byte[ 3 ],
    this->recently.read.byte[ 4 ], this->recently.read.byte[ 5 ],
    this->recently.read.byte[ 6 ], this->recently.read.byte[ 7 ] );
  if( sizeof( elomax_status_data ) <= count )
  {
    DBG_INFO( "user read status 0x%x\n", this->recently.status );
  }
  if( sizeof( elomax_monitor_data ) <= count )
    {
      DBG_INFO(
        "user read write-bytes=%02x %02x %02x %02x %02x %02x %02x %02x\n",
        this->recently.write.byte[ 0 ], this->recently.write.byte[ 1 ],
        this->recently.write.byte[ 2 ], this->recently.write.byte[ 3 ],
        this->recently.write.byte[ 4 ], this->recently.write.byte[ 5 ],
        this->recently.write.byte[ 6 ], this->recently.write.byte[ 7 ]  );
    }

  return retval;
}
/*----------------------------------------------------------------------------*/
static int
elomax_instance_read_connection_data(
  struct elomax_instance *this,
  char                   *buffer,
  size_t                  count )
{
  elomax_id_data  original = { ELOMAX_UNKNOWN_ID, "" };
  struct usb_device *device = interface_to_usbdev( this->interface );
  int connection_length = 0;
  int retval = 0;

  DBG_DEBUG( "retrieving type\n" );
  original.type = le16_to_cpu( device->descriptor.idProduct );
  DBG_DEBUG( "retrieving type returned type 0x%x\n", original.type );

  DBG_DEBUG( "retrieving connection\n" );
  connection_length =
    usb_make_path(
    device,
    original.connection,
    sizeof( elomax_connection_type ) - 2 );
  if( 0 <= connection_length )
  {
    original.connection[ connection_length ] = '\0';
  }
  else
  {
    DBG_DEBUG(
      "retrieving connection returned length %d\n",
      connection_length );
  }
  original.connection[ sizeof( elomax_connection_type ) - 1 ] = '\0';
  DBG_DEBUG(
    "retrieving connection returned %s\n",
    original.connection );

  retval = copy_to_user( buffer, &original, sizeof( original ) );
  DBG_INFO(
    "user read id solution_type %d, connection %s\n",
    original.type,
    original.connection );

  return retval;
}
/*----------------------------------------------------------------------------*/
static ssize_t
elomax_instance_read(
  struct file  *file,
  char         *buffer,
  size_t        count,
  loff_t       *ppos )
{
  BOOL                    ok = TRUE;
  struct elomax_instance *this = file->private_data;
  BOOL                    got_lock = FALSE;
  unsigned long irqflags = 0L;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "read from instance at %p\n", this );
    DBG_DEBUG( "read from interface at %p\n", this->interface );
  }
  if( ok )
  {
    DBG_DEBUG( "check bytes to read\n" );
    ok =
      ( sizeof( elomax_bare_data ) == count ) ||
      ( sizeof( elomax_status_data ) == count ) ||
      ( sizeof( elomax_monitor_data ) == count ) ||
      ( sizeof( elomax_id_data ) == count );
    debug_err(
     ok,
     "check bytes to read failed, need %lu, %lu, %lu or %lu bytes, "
     "user asked %d bytes\n",
     sizeof( elomax_bare_data ),
     sizeof( elomax_status_data ),
     sizeof( elomax_monitor_data ),
     sizeof( elomax_id_data ),
     STATIC_CAST( int, count ) );
  }
  if( ok )
  {
    DBG_DEBUG( "check if device is still connected\n" );
    spin_lock_irqsave( &this->instance_protect, irqflags );
    got_lock = TRUE;
    ok = NULL != this->interface;
    debug_err( ok, "check if device is still connected failed\n" );
  }
  if( ok )
  {
    int copy_status = 0;

    DBG_DEBUG( "copy %lu read bytes to user\n", count );
    switch( count )
    {
      case sizeof( elomax_bare_data ):
      case sizeof( elomax_status_data ):
      case sizeof( elomax_monitor_data ):
      {
        copy_status = elomax_instance_read_io_data( this, buffer, count );
        break;
      }
      case sizeof( elomax_id_data ):
      {
        copy_status =
          elomax_instance_read_connection_data( this, buffer, count );
        break;
      }
      default:
      {
        copy_status = -ENODEV;
        break;
      }
    }
    ok = 0 == copy_status;
    spin_unlock_irqrestore( &this->instance_protect, irqflags );
    got_lock = FALSE;
    debug_err(
      ok,
      "copy %lu read bytes to user failed with status=%d\n",
      count,
      copy_status );
  }

  if( ! ok )
  {
    DBG_DEBUG( "something failed, cleanup\n" );
    if( got_lock )
    {
      DBG_DEBUG( "release lock on this instance\n" );
      spin_unlock( &this->instance_protect );
    }
  }

  debug_leave;
  return ok ? count : -ENOMEM;
}
/*----------------------------------------------------------------------------*/
static ssize_t
elomax_instance_write(
  struct file  *file,
  const char   *user_buffer,
  size_t        count,
  loff_t       *ppos )
{
  BOOL                    ok = TRUE;
  struct elomax_instance *this = file->private_data;
  struct usb_device      *device = NULL;
  BOOL                    got_semaphore = FALSE;
  BOOL                    got_lock = FALSE;
  unsigned long irqflags = 0L;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "write to instance at %p\n", this );
    DBG_DEBUG( "write to interface at %p\n", this->interface );
  }
  if( ok )
  {
    DBG_DEBUG( "check bytes to write\n" );
    ok = sizeof( elomax_bare_data ) == count;
    debug_err(
     ok,
     "check bytes to write failed, need %lu, got %d bytes\n",
     sizeof( elomax_bare_data ),
     STATIC_CAST( int, count ) );
  }
  if( ok )
  {
    DBG_DEBUG( "wait for the writer-allowed semaphore\n" );
    down( &this->write_semaphore );
    got_semaphore = TRUE;
  }
  if( ok )
  {
    DBG_DEBUG( "get lock on this instance\n" );
    spin_lock_irqsave( &this->instance_protect, irqflags );
    got_lock = TRUE;
  }
  if( ok )
  {
    DBG_DEBUG( "check that interface is still connected\n" );
    DBG_DEBUG( "connected interface is at %p\n", this->interface );
    ok = NULL != this->interface;
    debug_err( ok, "check that interface is still connected failed\n" );
  }
  if( ok )
  {
    DBG_DEBUG( "check that device is still connected\n" );
    device = interface_to_usbdev( this->interface );
    DBG_DEBUG( "connected device is at %p\n", device );
    ok = NULL != device;
    debug_err( ok, "check that device is still connected failed\n" );
  }
  if( ok )
  {
    int copy_status = 0;

    DBG_DEBUG( "copy bytes from user to write-buffer\n" );
    copy_status =
      copy_from_user(
        this->write_urb->transfer_buffer,
        user_buffer,
        sizeof( elomax_bare_data ) );
    ok = 0 == copy_status;
    DBG_INFO(
      "user write data=%02x %02x %02x %02x %02x %02x %02x %02x\n",
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 0 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 1 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 2 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 3 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 4 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 5 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 6 ],
      STATIC_CAST( unsigned char *, this->write_urb->transfer_buffer )[ 7 ] );
    debug_err(
      ok,
      "copy bytes from user to write-buffer failed with status=%d\n",
      copy_status );
  }
  if( ok )
  {
    unsigned int pipe = 0;

    FILL_SETUP( this->write_setup_packet, 0x21, 0x09, 0x0200, 0x0000, 0x0008 );

    DBG_DEBUG( "fill write-urb with pipe and buffers\n" );
    pipe = usb_sndctrlpipe( device, 0 );
    DBG_DEBUG( "write-urb's pipe at %x\n", pipe );
    DBG_DEBUG(
      "write-setup-packet=%02x %02x %02x %02x %02x %02x %02x %02x\n",
      this->write_setup_packet[ 0 ], this->write_setup_packet[ 1 ],
      this->write_setup_packet[ 2 ], this->write_setup_packet[ 3 ],
      this->write_setup_packet[ 4 ], this->write_setup_packet[ 5 ],
      this->write_setup_packet[ 6 ], this->write_setup_packet[ 7 ]  );
    usb_fill_control_urb(
     this->write_urb,                 /* urb to fill */
     device,                          /* device to send it to */
     pipe,                            /* pipe to  send it over */
     this->write_setup_packet,        /* setup packet to send */
     this->write_urb->transfer_buffer,/* data buffer to send */
     sizeof( elomax_bare_data ),      /* data buffer size */
     elomax_instance_write_complete,  /* func to call at completion */
     this );                          /* context to store in the urb */
  }
  if( ok )
  {
    int submit_status = 0;

    DBG_DEBUG( "submit write-urb to usb-core\n" );
    submit_status = usb_submit_urb( this->write_urb, GFP_KERNEL );
    ok = 0 == submit_status;
    debug_err(
      ok,
      "submit write-urb to usb-core failed with status=%d\n",
      submit_status );
  }

  if( ok )
  {
    DBG_DEBUG( "release lock on this instance\n" );
    spin_unlock_irqrestore( &this->instance_protect, irqflags );
    got_lock = FALSE;
  }

  if( ! ok )
  {
    DBG_DEBUG( "something failed, cleanup\n" );
    if( got_lock )
    {
      DBG_DEBUG( "release lock on this instance\n" );
      spin_unlock( &this->instance_protect );
    }
    if( got_semaphore )
    {
      DBG_DEBUG( "release the writer-allowed semaphore" );
      up( &this->write_semaphore );
    }
  }

  debug_leave;
  return ok ? count : -ENOMEM;
}
/*----------------------------------------------------------------------------*/
static int
elomax_instance_open(
  struct inode  *inode,
  struct file   *file )
{
  BOOL ok = TRUE;
  struct elomax_instance *this = NULL;
  int retval = 0;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "retrieve instance for this inode's minor\n" );
    this = elomax_driver_instance_from_minor( iminor( inode ) );
    DBG_DEBUG( "this instance is at %p\n", this );
    ok = NULL != this;
    debug_err( ok, "retrieve instance for this inode's minor failed\n" );
    retval = -ENODEV;
  }
  if( ok )
  {
    DBG_SUMRY(
      "user opened instance %p for minor %d\n",
      this,
      iminor( inode ) );
    DBG_DEBUG( "store instance in file's private-data\n" );
    file->private_data = this;
    DBG_DEBUG( "this file uses this instance\n" );
    elomax_instance_use( this );
  }

  debug_leave;
  return ok ? 0 : retval;
}
/*----------------------------------------------------------------------------*/
static int
elomax_instance_close(
  struct inode  *inode,
  struct file   *file )
{
  BOOL ok = TRUE;
  struct elomax_instance *this = NULL;
  int retval = 0;

  debug_enter;

  if( ok )
  {
    DBG_DEBUG( "retrieve instance for this file\n" );
    this = file->private_data;
    DBG_DEBUG( "retrieved instance from file at %p\n", this );
    ok = NULL != this;
    debug_err( ok, "retrieve instance for this file failed\n" );
    retval = -ENODEV;
  }

  if( ok )
  {
    DBG_SUMRY(
      "user closed instance %p for minor %d\n",
      this,
      iminor( inode ) );
    DBG_DEBUG( "clear instance from file's private-data\n" );
    file->private_data = NULL;
    DBG_DEBUG( "this file doesn't need the instance anymore\n" );
    elomax_instance_destroy( this );
  }

  debug_leave;
  return ok ? 0 : retval;
}
/*----------------------------------------------------------------------------*/
