/*------------------------------------------------------------------------------
    elomaxdriver: A linux kernel driver to control http://www.elomax.nl
                  usb-io modules
    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef ELOMAX_INSTANCE_H
#define ELOMAX_INSTANCE_H
/*----------------------------------------------------------------------------*/
#include <linux/usb.h>
/*----------------------------------------------------------------------------*/
/**
  @file
  @brief
  This file contains the functions to construct and destruct an instance for an
  elomax unit. The operations an user application can execute on an instance,
  are described in @ref elomaxinstance-priv.h.

  @details
  An instance maintains the data for an elomax unit. It keeps the status (the
  usb urb's in progress, the locking, etc) for this unit.

  The functions elomax_instance_construct() and elomax_instance_destruct()
  respectively construct and destruct such an instance.
**/
/*----------------------------------------------------------------------------*/
struct elomax_instance;
/*----------------------------------------------------------------------------*/
/**
  @brief
  Constructs an instance for the elomax unit, which is by the usb-core
  identified by the given interface.

  @details
  This function checks the properties of the elomax unit if it is compatible. If
  so, this function reserves and initializes memory for the instance. This
  instance is then registered (a new device file is created, a minor is claimed)
  to the usb-core by calling @e usb_register_dev().

  Secondly this function reserves and starts a read urb (via usb_submit_urb()).
  This urb will catch the incoming data from the unit and deliver this via
  function elomax_instance_read_complete(). This completion function wil also
  restart the read urb, so that this instance is continuously aware of the
  latest data of the unit.

  @param [ in ] interface
  The identifier of the usb-core, which is unique for this elomax unit. The
  data this package wants to maintain for this unit (a struct elomax_instance)
  is attached to this interface.

  @return
  The constructed instance.
**/
struct elomax_instance *
elomax_instance_construct( struct usb_interface *interface );
/*----------------------------------------------------------------------------*/
/**
  @brief
  Destructs the given elomax_instance.

  @details
  This function deregisters the given instance from the usb-core by calling
  @e usb_deregister_dev(). It then uncouples the instance from its interface
  and announces it for destruction.

  The destruction announcement is done by decrementing the
  elomax_instance::instance_smartptr. This smartptr (reference counter) controls
  the real destruction of the instance.

  The usb-core takes care that the completion-functiosn for the urb's in
  progress are called. This will stop (and not restart) the i/o to the device.

  @param[ in ]  this
  The elomax_instance which must be destroyed.
**/
void
elomax_instance_destruct( struct elomax_instance *this );
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
