/*------------------------------------------------------------------------------
    MockC:    a framework to implement mocking when unit-testing

    Copyright (C) 2011  programming <at> kogro org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
#ifndef MOCK_H
#define MOCK_H
/*----------------------------------------------------------------------------*/
/*
  This file is part of the testing framework, it is a mocking extension to
  CUnit. The ideas for this testing framework are based on MockPP
  (http://mockpp.sourceforge.net/index-en.html).

  I N T R O D U C T I O N
  -----------------------
  Testing is verifying that your 'module under test', when triggered, interacts
  with its environment and delivers the result as expected.
  In this sentence, the three major parts in testing are mentioned:
  1) The trigger-part
     This is the part that calls the 'module under test' with defined
     parameters, so that the 'module under test' is submitted to a defined
     scenario.
  2) The 'module under test'
     This is the part that is being tested. It is called, interacts with its
     environment, and delivers a result. The interaction and the result is
     verified to be as expected.
  3) The 'environment' of the 'module under test'
     This is where this framework comes in. This mock-framework simulates the
     normal environment for the 'module under test', whereby it during the test
     verifies that it (the environment) is triggered as expected.

  The cooperation between these 3 items can be depicted as shown below:

        +-------+                                  +------+
        | CUnit |           expectations           | Mock |
        |       |  ----------------------------->  |      |
        |       |                                  |      |
        |       |            +--------+            |      |
        |       |  trigger   | Module |    mock    |      |
        |       |  ------->  | Under  |  <-------  |      |
        |       |            | Test   |            |      |
        |       |            |        |            |      |
        |       |            +--------+            |      |
        |       |                                  |      |
        |       |              verify              |      |
        |       |  <-----------------------------  |      |
        |       |                                  |      |
        +-------+                                  +------+

  S C O P E
  ---------
  This mocking framework is written to cooperate with CUnit.

  The 'module under test' is (a couple of) C-functions which, when in their
  normal program, call other C-functions. When the 'module under test' is
  isolated from their normal program and placed under CUnit, the Mock-functions
  take the place of the environment C-functions.

  P R I N C I P L E   O F   O P E R A T I O N
  -------------------------------------------
  In the beginning, the testsuite of CUnit announces the expectations for the
  Mock. CUnit tells the Mock, which parameters to expect and which return value
  to deliver.
  Next the testsuite of CUnit triggers the 'module under test', which interacts
  with its environment (and calls the Mock). The Mock notes the call and
  delivers the return value as announced during the expect.
  At the end of the testsuite the Mock reports whether all invocations were as
  expected.

  From the 'module under test' point of view, the Mock is a C function with a
  known signature (a certain number and type of parameters and a return type).

  From the CUnit point of view, the Mock has a double interface. First it is
  a C function where CUnit can set the expectations and tell the Mock that a
  new test is started. Secondly the Mock is a C function where CUnit can ask
  whether the invocation was successful.

  When the module under test in its environment needs a
  'int Func( char, float )' function, the Mock framework implements a:
    int  Func( char, float )    // the actual mocked function
    void Func_reset( void )     // to tell the Mock to restart for a new test
    void Func_expect(           // to tell the Mock the expectations
      int,                         // expected return value
      (int *)( char, char ),       // comparison function for 1st parameter
      char,                        // expected value for 1st parameter
      (int *)( float, float ),     // comparison function for 2nd parameter
      float )                      // expected value for 2nd parameter
    int  Func_verify( void )    // to ask the Mock for the result

  Internally, the Mock stores all expectations in an array and keeps a counter
  (counting the number of invocations) up to date. When called, the Mock uses
  the counter as an index in the array to find the expectation for this call.
  It uses then the comparison functions to check if the invocation was as
  expected and stores the result in the array and returns the return value to
  the module under test. When asked to verify, the Mock returns the summary of
  all 'CalledOk' results in the array.

  I M P L E M E N T A T I O N
  ---------------------------
  The implementation is driven by a number of constraints:
  - no limitations on the 'module under test' or 'environment to mock'
  - creating a testsuite using Mock must be as easy as possible
  - creating a new Mock must be as easy as possible

  According the first constraint, the Mock framework must be capable to mock
  any signature of environment function. This signature is only known at
  compile-time, not at design-time. To solve this, the Mock framework heavily
  relies on the preprocessor to create the necessary functions.

  There is a Mock file per number of parameters in the function. This Mock
  header file defines a number of macro's for the test-builder:
    MOCK_2                   // define the declaration for an environment
                             //   function with return type and 2 parameters
    MOCK_VOID_2              // define the declaration for an environment
                             //   function with void return and 2 parameters
    MOCKIMPL_2               // define the implementation for an environment
                             //   function with return type and 2 parameters
    MOCKIMPL_VOID_2          // define the implementation for an environment
                             //   function with void return and 2 parameters

  To extend the Mock framework for environment functions with another number of
  parameters in their prototype, this mock2.h can be copied and the
  implementation of the macro's changed so, that they are suitable for another
  number of parameters.

  The implementation of these macro's is based on more general mock macro's.
  These macro's are implemented in mock.h (this file). Unless the functionality
  of the Mock framework is changed, this file needs no maintenance.

  Graphically the design of the mock framework looks like:


                                          +-------+
              mock                        | Mock  |
              base                        +-------+
                                              ^
                                              |
                                  +-----------+----------+
                                  |           |          |
                              +--------+  +-------+  +--------+
              mock            | Mock0  |  | Mock1 |  | Mock2  |
              framework       +--------+  +-------+  +--------+
                                ^             ^        ^    ^
                                |             |        |    |
                                +----------------------+    |
                                |             |             |
                                |             +-------------+
                                |                           |
                              +--------+  +-------+  +--------+
              mock            | Unit 1 |  | Unit2 |  | Unit 3 |
              objects         | Mock   |  | Mock  |  | Mock   |
                              +--------+  +-------+  +--------+
                                  ^          ^           ^
  +--------+                      |          |           |
  | Module |                      |          |           |
  | Under  | ---------------------+----------+-----------+
  | Test   |
  +--------+

  The 'mock objects' are the instantiations of the environment functions. These
  files are owned by the test-builder, who 'creates' the functions to be
  mocked.

  The 'mock framework' are the macros that create mock functions with a defined
  number of parameters, with and without return type. These files are owned
  by the mock-builder, who creates a new file when a 'module under test' needs
  more parameters in a function signature.

  The 'mock base' are the base macros for the mock-framework. These macro's
  define the functionality of the framework. The mock-builder maintains this
  file when the functionality needs to be changed.

*/
/*----------------------------------------------------------------------------*/
#include <stdio.h>
#include "CUnit/Basic.h"
/*----------------------------------------------------------------------------*/
#define MOCK_MAX_EXPECTATIONS              ( 100 )
/*----------------------------------------------------------------------------*/
#define MOCK_COMPARE( Type )               int (*)( Type, Type )
#define MOCK_NAMED_COMPARE( Name, Type )   int (*Name)( Type, Type )
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCK( TReturn, Func )                                                 \
  int Func##_verify( void );                                                  \
  void Func##_reset( void );
/*----------------------------------------------------------------------------*/
/*
    I M P L E M E N T A T I O N
*/
/*----------------------------------------------------------------------------*/
/*  Function */
#define MOCKFUNC_BEGIN( Func )                                                \
  {                                                                           \
    int MoreCallsThenExpectations =                                           \
      ( Func##Expectance.CallNr < Func##Expectance.UsedExpectations );        \
    if( ! MoreCallsThenExpectations )                                         \
    {                                                                         \
      printf( "Mock %s: call %d, more calls then expectations\n",             \
        #Func,                                                                \
        Func##Expectance.CallNr );                                            \
    }                                                                         \
    CU_ASSERT( MoreCallsThenExpectations );                                   \
    Func##Expectance.Expectation[ Func##Expectance.CallNr ].IsCalled = 1;     \
    Func##Expectance.Expectation[ Func##Expectance.CallNr ].CalledOk = 1;
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKFUNC_TEST_PARAM( Func, TParam, Nr )                               \
  if( ! MoreCallsThenExpectations )                                           \
  {                                                                           \
    printf( "called with " );                                                 \
    MockShow##TParam( Param##Nr );                                            \
    printf( "\n" );                                                           \
  }                                                                           \
  if(                                                                         \
    ( NULL !=                                                                 \
        Func##Expectance.Expectation[ Func##Expectance.CallNr ]               \
          .TestParam##Nr ) )                                                  \
  {                                                                           \
    if(                                                                       \
      ! Func##Expectance.Expectation[ Func##Expectance.CallNr ]               \
          .TestParam##Nr(                                                     \
            Param##Nr,                                                        \
            Func##Expectance.Expectation[ Func##Expectance.CallNr ]           \
              .Param##Nr ) )                                                  \
    {                                                                         \
      int ExpectationNotMet =                                                 \
      Func##Expectance.Expectation[ Func##Expectance.CallNr ].CalledOk = 0;   \
      if( ! ExpectationNotMet )                                               \
      {                                                                       \
        printf( "Mock %s: call %d, expectation not met\n",                    \
          #Func,                                                              \
          Func##Expectance.CallNr );                                          \
        printf( "expected " );                                                \
        MockShow##TParam(                                                     \
          Func##Expectance.Expectation[ Func##Expectance.CallNr ]             \
            .Param##Nr );                                                     \
        printf( ", got " );                                                   \
        MockShow##TParam( Param##Nr );                                        \
        printf( "\n" );                                                       \
      }                                                                       \
      CU_ASSERT( ExpectationNotMet );                                         \
    }                                                                         \
  }  
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKFUNC_END( Func )                                                  \
    {                                                                         \
      int MoreCallsThenReturnValues =                                         \
        ( Func##Expectance.CallNr < Func##Expectance.UsedExpectations );      \
      if( ! MoreCallsThenReturnValues )                                       \
      {                                                                       \
        printf( "Mock %s: call %d, more calls then expected return values\n", \
          #Func,                                                              \
          Func##Expectance.CallNr );                                          \
      }                                                                       \
      CU_ASSERT( MoreCallsThenReturnValues );                                 \
    }                                                                         \
    return Func##Expectance.Expectation[ Func##Expectance.CallNr++ ].Return;  \
  }
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKFUNC_VOID_END( Func )                                             \
    ++Func##Expectance.CallNr;                                                \
  }
/*----------------------------------------------------------------------------*/
/*  Expectation-variable */
#define MOCKVARIABLE_VOID_BEGIN( Func )                                       \
  struct s##Func##Expect                                                      \
  {
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKVARIABLE_BEGIN( TReturn, Func )                                   \
  MOCKVARIABLE_VOID_BEGIN( Func )                                             \
    TReturn    Return;
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKVARIABLE_ENTRY( Nr, Type )                                        \
    MOCK_NAMED_COMPARE( TestParam##Nr, Type );                                \
    Type       Param##Nr;
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKVARIABLE_END( Func )                                              \
    int        IsCalled;                                                      \
    int        CalledOk;                                                      \
  };                                                                          \
  typedef struct s##Func##Expect  s##Func##Expect;                            \
  struct s##Func##Expectance                                                  \
  {                                                                           \
    s##Func##Expect Expectation[ MOCK_MAX_EXPECTATIONS ];                     \
    int             UsedExpectations;                                         \
    int             CallNr;                                                   \
  };                                                                          \
  typedef struct s##Func##Expectance  s##Func##Expectance;                    \
  static s##Func##Expectance          Func##Expectance = { { }, 0, 0 };
/*----------------------------------------------------------------------------*/
#define MOCKRESET( Func )                                                     \
  void                                                                        \
  Func##_reset( void )                                                        \
  {                                                                           \
    Func##Expectance.CallNr = 0;                                              \
    Func##Expectance.UsedExpectations = 0;                                    \
  }
/*----------------------------------------------------------------------------*/
/*  Expectation-function */
#define MOCKEXPECT_BEGIN( Func )                                              \
    {                                                                         \
      int ToManyExpectationsForFramework =                                    \
        ( MOCK_MAX_EXPECTATIONS > Func##Expectance.UsedExpectations );        \
      if( ! ToManyExpectationsForFramework )                                  \
      {                                                                       \
        printf( "Mock %s: call %d, too many expectations for framework\n",    \
          #Func,                                                              \
          Func##Expectance.CallNr );                                          \
      }                                                                       \
      CU_ASSERT( ToManyExpectationsForFramework );                            \
    }
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKEXPECT_RETURN( TReturn, Func )                                    \
    Func##Expectance.Expectation[ Func##Expectance.UsedExpectations ]         \
      .Return = Return;
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKEXPECT_PARAM( Func, Nr )                                          \
    Func##Expectance.Expectation[ Func##Expectance.UsedExpectations ]         \
      .TestParam##Nr = TestParam##Nr;                                         \
    Func##Expectance.Expectation[ Func##Expectance.UsedExpectations ]         \
      .Param##Nr = Param##Nr
/* --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  --  -- */
#define MOCKEXPECT_END( Func )                                                \
    Func##Expectance.Expectation[ Func##Expectance.UsedExpectations ]         \
      .IsCalled = 0;                                                          \
    Func##Expectance.Expectation[ Func##Expectance.UsedExpectations ]         \
      .CalledOk = 1;                                                          \
    ++Func##Expectance.UsedExpectations
/*----------------------------------------------------------------------------*/
#define MOCKVERIFY( Func )                                                    \
  int                                                                         \
  Func##_verify( void )                                                       \
  {                                                                           \
    int l_AllOk = 1;                                                          \
    int  i = 0;                                                               \
                                                                              \
    for( i = 0; i < Func##Expectance.UsedExpectations; ++i )                  \
    {                                                                         \
      if( ! Func##Expectance.Expectation[ i ].IsCalled )                      \
      {                                                                       \
        printf( "Mock %s: expectation %d (out of %d) was not called\n",       \
          #Func, i, Func##Expectance.UsedExpectations );                      \
      }                                                                       \
      if( ! Func##Expectance.Expectation[ i ].CalledOk )                      \
      {                                                                       \
        printf( "Mock %s: expectation %d (out of %d) was not met\n",          \
          #Func, i, Func##Expectance.UsedExpectations );                      \
      }                                                                       \
      l_AllOk =                                                               \
        l_AllOk &&                                                            \
        Func##Expectance.Expectation[ i ].IsCalled &&                         \
        Func##Expectance.Expectation[ i ].CalledOk;                           \
    }                                                                         \
                                                                              \
    return l_AllOk;                                                           \
  }
/*----------------------------------------------------------------------------*/
#endif
/*----------------------------------------------------------------------------*/
